﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Diagnostics;
using System.Linq;
using System.Text;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessAHOBPR.MilitaryInfoService;
using System.Reflection;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis
{
    public class EligibilityInfoImporter
    {

        public VeteranStatus GetVeteranStatus(MilitaryServiceEligibility serviceEligibilityInstance)
        {
            return serviceEligibilityInstance?.veteranStatus ?? new VeteranStatus();
        }

        public ICollection<MilitaryServiceEpisodeEligibilityData> GetServiceEpisodes(MilitaryServiceEligibility serviceEligibilityInstance)
        {
            var serviceEpisodes = new List<MilitaryServiceEpisodeEligibilityData>();

            if (ContainerAndListAreNotNull(serviceEligibilityInstance, serviceEligibilityInstance.militaryServiceEpisodes))
            {
                serviceEpisodes = serviceEligibilityInstance.militaryServiceEpisodes.ToList();
            }
            return serviceEpisodes;
        }

        public bool ContainerAndListAreNotNull<T, V>(T value, V internalList)
        {
            bool isNotNull = false;
            if (value != null)
            {
                isNotNull |= (internalList != null);
            }
            return isNotNull;
        }
        public ICollection<MilitaryServiceEligibility> GetMilitaryServiceEligibilityInfo(string edipi, eMISMilitaryInformationSerivcePortTypesClient client)
        {
            var serviceEligibilityInstances = new List<MilitaryServiceEligibility>();
            if (client == null)
                throw new ArgumentNullException(nameof(client));

            var headerInfo = new InputHeaderInfo();
            inputEdiPiOrIcn inputEdipi = InputEdipiOrIcnCreator.Create(edipi);
            eMISmilitaryServiceEligibilityResponseType response = client.getMilitaryServiceEligibilityInfo(ref headerInfo, inputEdipi);
            serviceEligibilityInstances = response.militaryServiceEligibility.ToList();
            return serviceEligibilityInstances;
        }

        public void ExtractDataFromServiceEligibilityInstances(ICollection<MilitaryServiceEligibility> serviceEligibilityInstances, REGISTRANT registrant)
        {
            if (serviceEligibilityInstances == null)
                throw new ArgumentNullException(nameof(serviceEligibilityInstances));
            if (registrant == null)
                throw new ArgumentNullException(nameof(registrant));
            foreach (var instance in serviceEligibilityInstances)
            {
                try
                {
                    //Get Service Episodes and add them to the Registrant object
                    ProcessServiceEpisodes(instance, registrant);
                    Get911IndicatorsFromVeteranStatus(instance, registrant);
                }
                catch (ArgumentNullException ex)
                {
                    AHOBPRLogger.LogErrorMessage("ArgumentNullException",
                                            this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name,
                                            "Null argument: " + ex.ParamName);
                }
            }
        }

        private void Get911IndicatorsFromVeteranStatus(MilitaryServiceEligibility serviceEligibilityInstance, REGISTRANT registrant)
        {
            if (serviceEligibilityInstance == null)
                throw new ArgumentNullException(nameof(serviceEligibilityInstance));
            registrant.PRE_911_DEPLOYMENT = serviceEligibilityInstance.veteranStatus.pre911DeploymentIndicator;
            registrant.POST_911_DEPLOYMENT = serviceEligibilityInstance.veteranStatus.post911DeploymentIndicator;
        }

        /// <summary>
        /// Iterate through service eipsodes to extract a service episode entry
        /// </summary>
        /// <param name="serviceEligibilityInstance"></param>
        /// <param name="registrant"></param>
        public void ProcessServiceEpisodes(MilitaryServiceEligibility serviceEligibilityInstance, REGISTRANT registrant)
        {
            if (serviceEligibilityInstance == null)
                throw new ArgumentNullException(nameof(serviceEligibilityInstance));
            if (registrant == null)
                throw new ArgumentNullException(nameof(registrant));

            MarkExistingEpisodesForDeletion(registrant);
            ICollection<MilitaryServiceEpisodeEligibilityData> eps = GetServiceEpisodes(serviceEligibilityInstance);
            foreach (var episode in eps)
            {
                try
                {
                    REGISTRANT_SERVICE_EPISODE regEpisode = CreateRegistrantServiceEpisodeEntry(registrant.REGISTRANT_ID, episode);
                    registrant.REGISTRANT_SERVICE_EPISODEs.Add(regEpisode);
                    registrant.REGISTRANT_SERVICE_EPISODEs.Last().SetAsInsertOnSubmit();
                }
                catch (ArgumentNullException ex)
                {
                    AHOBPRLogger.LogErrorMessage("ArgumentNullException",
                                             this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name,
                                             "CreateRegistrantServiceEpisodeEntry method given null argument: " + ex.ParamName);
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrant"></param>
        private static void MarkExistingEpisodesForDeletion(REGISTRANT registrant)
        {
            foreach (REGISTRANT_SERVICE_EPISODE registrantServiceEpisode in registrant.REGISTRANT_SERVICE_EPISODEs)
            {
                registrantServiceEpisode.SetAsDeleteOnSubmit();
            }
        }

        /// <summary>
        /// Create a DB REGISTRANT_SERVICE_EPISODE object from the 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="episode"></param>
        public REGISTRANT_SERVICE_EPISODE CreateRegistrantServiceEpisodeEntry(int registrantId, MilitaryServiceEpisodeEligibilityData episode)
        {
            if (episode == null)
                throw new ArgumentNullException(nameof(episode));
            var serviceEpisode = new REGISTRANT_SERVICE_EPISODE();
            serviceEpisode.REGISTRANT_ID = registrantId;
            serviceEpisode.SERVICE_EPISODE_TERMINATION_REASON = episode.narrativeReasonForSeparationCode;

                serviceEpisode.SERVICE_EPISODE_START_DATE = episode.serviceEpisodeStartDate.ToUniversalTime();


                serviceEpisode.SERVICE_EPISODE_END_DATE = episode.serviceEpisodeEndDate.ToUniversalTime();

            int branchOfServiceId = AHOBPRShared.GetStdBranchOfServiceId(episode.branchOfServiceCode);
            if (branchOfServiceId > 0)
            {
                serviceEpisode.STD_BRANCH_OF_SERVICE_ID = branchOfServiceId;
            }
            serviceEpisode.BRANCH_OF_SERVICE_TEXT = episode.branchOfServiceCode;

            return serviceEpisode;
        }

    }
}
